<?php
/*
 * Configuración general del sistema (Hito 4).
 *
 * Permite gestionar los medios de pago y la configuración de los tickets (encabezado, pie, logo y prueba de impresión).
 */

require_once __DIR__ . '/includes/functions.php';
requireLogin();
// Se requiere permiso para ver configuraciones
requirePermission('settings', 'view');

$pdo = getPDO();
if (!$pdo) {
    echo '<div class="container mt-5"><div class="alert alert-danger">No se pudo conectar a la base de datos.</div></div>';
    exit;
}

// Asegurar la tabla de medios de pago
ensurePaymentTables($pdo);

// -------------------------
// Gestión de medios de pago
// -------------------------
$pmAction = $_GET['pm_action'] ?? '';
$pmId     = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$pmError  = '';

// Eliminar medio de pago
if ($pmAction === 'delete' && $pmId > 0) {
    requirePermission('settings', 'delete');
    deletePaymentMethod($pdo, $pmId);
    redirect('settings.php');
}

// Crear o actualizar medio de pago
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['pm_action'])) {
    $pmFormAction = $_POST['pm_action'];
    // Solo los usuarios con permiso de crear o editar pueden ejecutar acciones
    if ($pmFormAction === 'create') {
        requirePermission('settings', 'create');
    } elseif ($pmFormAction === 'edit') {
        requirePermission('settings', 'edit');
    }
    $name    = trim($_POST['pm_name'] ?? '');
    $active  = isset($_POST['pm_active']) ? true : false;
    $discount = (float)($_POST['pm_discount'] ?? 0);
    if ($name === '') {
        $pmError = 'El nombre del medio de pago es obligatorio.';
    } elseif ($discount < 0 || $discount > 100) {
        $pmError = 'El porcentaje de descuento debe ser un valor entre 0 y 100.';
    } else {
        if ($pmFormAction === 'create') {
            createPaymentMethod($pdo, $name, $active, $discount);
        } elseif ($pmFormAction === 'edit' && $pmId > 0) {
            updatePaymentMethod($pdo, $pmId, $name, $active, $discount);
        }
        redirect('settings.php');
    }
}

// -------------------------
// Gestión de tickets
// -------------------------

$ticketHeader  = getSetting($pdo, 'ticket_header') ?? '';
$ticketFooter  = getSetting($pdo, 'ticket_footer') ?? '';
$ticketLogo    = getSetting($pdo, 'ticket_logo') ?? '';
$ticketMessage = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ticket_action']) && $_POST['ticket_action'] === 'save') {
    // Requiere permiso de edición de configuraciones
    requirePermission('settings', 'edit');
    $headerInput = trim($_POST['ticket_header'] ?? '');
    $footerInput = trim($_POST['ticket_footer'] ?? '');
    // Guardar encabezado y pie
    saveSetting($pdo, 'ticket_header', $headerInput);
    saveSetting($pdo, 'ticket_footer', $footerInput);
    // Procesar carga de logo
    if (isset($_FILES['ticket_logo']) && $_FILES['ticket_logo']['error'] === UPLOAD_ERR_OK) {
        $tmpName = $_FILES['ticket_logo']['tmp_name'];
        $origName = $_FILES['ticket_logo']['name'];
        $type     = mime_content_type($tmpName);
        $allowed  = ['image/jpeg','image/png','image/webp','image/jpg'];
        if (in_array($type, $allowed)) {
            $uploadDir = __DIR__ . '/uploads';
            if (!is_dir($uploadDir)) {
                mkdir($uploadDir, 0777, true);
            }
            $ext      = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
            $newName  = 'logo_' . time() . '.' . $ext;
            $destPath = $uploadDir . '/' . $newName;
            if (move_uploaded_file($tmpName, $destPath)) {
                // Guardar ruta relativa
                $relative = 'uploads/' . $newName;
                saveSetting($pdo, 'ticket_logo', $relative);
                $ticketLogo = $relative;
            }
        }
    }
    $ticketHeader  = $headerInput;
    $ticketFooter  = $footerInput;
    $ticketMessage = 'Configuración de tickets guardada correctamente.';
}

require_once __DIR__ . '/layout.php';
renderHeader('Configuraciones', 'settings');
?>
<div class="container-fluid">
    <h1 class="mb-4">Configuraciones Generales</h1>

    <!-- Medios de pago -->
    <div class="card mb-4">
        <div class="card-header d-flex justify-content-between align-items-center">
            <h5 class="mb-0">Medios de pago</h5>
            <?php if ($pmAction !== 'create' && !($pmAction === 'edit' && $pmId > 0)): ?>
                <?php if (hasPermission('settings', 'create')): ?>
                    <a href="settings.php?pm_action=create" class="btn btn-success btn-sm">Nuevo medio</a>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        <div class="card-body">
            <?php if ($pmAction === 'create' || ($pmAction === 'edit' && $pmId > 0)): ?>
                <?php
                $editing = ($pmAction === 'edit');
                $pmData = ['name' => '', 'active' => 1, 'discount_percent' => 0];
                if ($editing) {
                    $pmRow = getPaymentMethodById($pdo, $pmId);
                    if (!$pmRow) {
                        echo '<div class="alert alert-danger">Medio de pago no encontrado.</div>';
                    } else {
                        $pmData = $pmRow;
                    }
                }
                ?>
                <h5 class="mb-3"><?php echo $editing ? 'Editar medio de pago' : 'Nuevo medio de pago'; ?></h5>
                <?php if ($pmError): ?>
                    <div class="alert alert-danger"><?php echo escape($pmError); ?></div>
                <?php endif; ?>
                <form method="post" action="">
                    <input type="hidden" name="pm_action" value="<?php echo $editing ? 'edit' : 'create'; ?>">
                    <div class="mb-3">
                        <label for="pm_name" class="form-label">Nombre</label>
                        <input type="text" class="form-control" id="pm_name" name="pm_name" required value="<?php echo escape($pmData['name']); ?>">
                    </div>
                    <div class="mb-3">
                        <label for="pm_discount" class="form-label">% de descuento</label>
                        <input type="number" step="0.01" min="0" max="100" class="form-control" id="pm_discount" name="pm_discount" value="<?php echo escape($pmData['discount_percent']); ?>">
                    </div>
                    <div class="form-check form-switch mb-3">
                        <input class="form-check-input" type="checkbox" id="pm_active" name="pm_active" value="1" <?php echo (!empty($pmData['active']) ? 'checked' : ''); ?>>
                        <label class="form-check-label" for="pm_active">Activo</label>
                    </div>
                    <button type="submit" class="btn btn-primary"><?php echo $editing ? 'Actualizar' : 'Crear'; ?></button>
                    <a href="settings.php" class="btn btn-secondary">Cancelar</a>
                </form>
            <?php else: ?>
                <div class="table-responsive">
                    <table class="table table-striped table-bordered align-middle">
                        <thead>
                        <tr>
                            <th>ID</th>
                            <th>Nombre</th>
                            <th>Descuento (%)</th>
                            <th>Estado</th>
                            <th>Acciones</th>
                        </tr>
                        </thead>
                        <tbody>
                        <?php foreach (getPaymentMethods($pdo) as $pm): ?>
                            <tr>
                                <td><?php echo (int)$pm['id']; ?></td>
                                <td><?php echo escape($pm['name']); ?></td>
                                <td><?php echo number_format((float)$pm['discount_percent'], 2); ?></td>
                                <td>
                                    <?php echo ($pm['active'] ? '<span class="badge bg-success">Activo</span>' : '<span class="badge bg-secondary">Inactivo</span>'); ?>
                                </td>
                                <td>
                                    <?php if (hasPermission('settings','edit')): ?>
                                        <a href="settings.php?pm_action=edit&amp;id=<?php echo (int)$pm['id']; ?>" class="btn btn-sm btn-primary">Editar</a>
                                    <?php endif; ?>
                                    <?php if (hasPermission('settings','delete')): ?>
                                        <a href="settings.php?pm_action=delete&amp;id=<?php echo (int)$pm['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar este medio de pago?');">Eliminar</a>
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <!-- Configuración de tickets -->
    <div class="card mb-4">
        <div class="card-header">
            <h5 class="mb-0">Configuración de tickets</h5>
        </div>
        <div class="card-body">
            <?php if ($ticketMessage): ?>
                <div class="alert alert-success"><?php echo escape($ticketMessage); ?></div>
            <?php endif; ?>
            <form method="post" action="" enctype="multipart/form-data">
                <input type="hidden" name="ticket_action" value="save">
                <div class="mb-3">
                    <label for="ticket_header" class="form-label">Encabezado del ticket</label>
                    <textarea class="form-control" id="ticket_header" name="ticket_header" rows="2"><?php echo escape($ticketHeader); ?></textarea>
                </div>
                <div class="mb-3">
                    <label for="ticket_footer" class="form-label">Pie del ticket</label>
                    <textarea class="form-control" id="ticket_footer" name="ticket_footer" rows="2"><?php echo escape($ticketFooter); ?></textarea>
                </div>
                <div class="mb-3">
                    <label for="ticket_logo" class="form-label">Logo</label>
                    <?php if ($ticketLogo): ?>
                        <div class="mb-2">
                            <img src="<?php echo baseUrl($ticketLogo); ?>" alt="Logo actual" class="img-fluid" style="max-height: 100px;">
                        </div>
                    <?php endif; ?>
                    <input type="file" class="form-control" id="ticket_logo" name="ticket_logo" accept="image/*">
                    <div class="form-text">Seleccione una imagen (PNG/JPEG/WebP). Si no desea cambiar el logo, deje vacío.</div>
                </div>
                <button type="submit" class="btn btn-primary">Guardar configuración</button>
                <a href="ticket_test.php" target="_blank" class="btn btn-secondary">Impresión de prueba</a>
            </form>
        </div>
    </div>
</div>
<?php
renderFooter();