<?php
/*
 * Gestión de roles: listado, creación, edición y eliminación.
 * Permite asignar permisos granulares a cada rol sobre los módulos del sistema.
 */

require_once __DIR__ . '/includes/functions.php';
requireLogin();
// Comprobar permiso de vista para roles
requirePermission('roles', 'view');

$pdo = getPDO();
if (!$pdo) {
    echo '<div class="container mt-5"><div class="alert alert-danger">No se pudo conectar a la base de datos.</div></div>';
    exit;
}
// Asegurar tablas de roles y permisos
ensureRoleTables($pdo);

// Variables de control
$action = $_GET['action'] ?? 'list';
$roleId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

// Manejar acciones
if ($action === 'delete' && $roleId > 0) {
    requirePermission('roles', 'delete');
    deleteRole($pdo, $roleId);
    redirect('roles.php');
}

// Procesamiento de formulario para crear/editar
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name        = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $permissions = [];
    foreach (getModules() as $module) {
        $permissions[$module] = [
            'view'   => isset($_POST['perm'][$module]['view']) ? true : false,
            'create' => isset($_POST['perm'][$module]['create']) ? true : false,
            'edit'   => isset($_POST['perm'][$module]['edit']) ? true : false,
            'delete' => isset($_POST['perm'][$module]['delete']) ? true : false,
        ];
    }
    // Evitar nombre vacío
    if ($name === '') {
        $errorMessage = 'El nombre del rol es obligatorio.';
    } else {
        if ($action === 'create') {
            requirePermission('roles', 'create');
            createRole($pdo, $name, $description, $permissions);
            redirect('roles.php');
        } elseif ($action === 'edit' && $roleId > 0) {
            // No permitir modificar SuperAdmin
            $role = getRoleById($pdo, $roleId);
            if ($role && strtolower($role['name']) === 'superadmin') {
                $errorMessage = 'No se puede modificar el rol SuperAdmin.';
            } else {
                requirePermission('roles', 'edit');
                updateRole($pdo, $roleId, $name, $description, $permissions);
                redirect('roles.php');
            }
        }
    }
}

require_once __DIR__ . '/layout.php';
renderHeader('Roles', 'roles');

?>
<div class="container-fluid">
    <?php if ($action === 'create' || ($action === 'edit' && $roleId > 0)): ?>
        <?php
        // Preparar datos para editar
        $editing = ($action === 'edit');
        $roleData = ['name' => '', 'description' => '', 'permissions' => []];
        if ($editing) {
            $role = getRoleById($pdo, $roleId);
            if (!$role) {
                echo '<div class="alert alert-danger">Rol no encontrado.</div>';
            } else {
                $roleData['name'] = $role['name'];
                $roleData['description'] = $role['description'];
                $roleData['permissions'] = getRolePermissions($pdo, $roleId);
            }
        } else {
            // Permisos por defecto: todos a false
            foreach (getModules() as $m) {
                $roleData['permissions'][$m] = [
                    'view'   => false,
                    'create' => false,
                    'edit'   => false,
                    'delete' => false,
                ];
            }
        }
        ?>
        <h1 class="mb-4"><?php echo $editing ? 'Editar rol' : 'Crear rol'; ?></h1>
        <?php if (!empty($errorMessage)): ?>
            <div class="alert alert-danger"><?php echo escape($errorMessage); ?></div>
        <?php endif; ?>
        <form method="post" action="">
            <div class="mb-3">
                <label for="name" class="form-label">Nombre</label>
                <input type="text" class="form-control" id="name" name="name" required value="<?php echo escape($roleData['name']); ?>">
            </div>
            <div class="mb-3">
                <label for="description" class="form-label">Descripción</label>
                <input type="text" class="form-control" id="description" name="description" value="<?php echo escape($roleData['description']); ?>">
            </div>
            <h5>Permisos</h5>
            <div class="table-responsive">
                <table class="table table-bordered">
                    <thead>
                    <tr>
                        <th>Módulo</th>
                        <th>Ver</th>
                        <th>Crear</th>
                        <th>Editar</th>
                        <th>Eliminar</th>
                    </tr>
                    </thead>
                    <tbody>
                    <?php foreach (getModules() as $module): ?>
                        <tr>
                            <td><?php echo ucfirst($module); ?></td>
                            <?php foreach (['view','create','edit','delete'] as $act): ?>
                                <td class="text-center">
                                    <input type="checkbox" name="perm[<?php echo $module; ?>][<?php echo $act; ?>]" value="1"<?php echo (!empty($roleData['permissions'][$module][$act])) ? ' checked' : ''; ?><?php echo (strtolower($roleData['name']) === 'superadmin') ? ' disabled' : ''; ?>>
                                </td>
                            <?php endforeach; ?>
                        </tr>
                    <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <button type="submit" class="btn btn-primary mt-2"><?php echo $editing ? 'Actualizar' : 'Crear'; ?></button>
            <a href="roles.php" class="btn btn-secondary mt-2">Cancelar</a>
        </form>
    <?php else: ?>
        <h1 class="mb-4">Roles</h1>
        <?php if (hasPermission('roles','create')): ?>
            <a href="roles.php?action=create" class="btn btn-success mb-3">Nuevo rol</a>
        <?php endif; ?>
        <div class="table-responsive">
            <table class="table table-striped table-bordered">
                <thead>
                <tr>
                    <th>ID</th>
                    <th>Nombre</th>
                    <th>Descripción</th>
                    <th>Acciones</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach (getRoles($pdo) as $r): ?>
                    <tr>
                        <td><?php echo (int)$r['id']; ?></td>
                        <td><?php echo escape($r['name']); ?></td>
                        <td><?php echo escape($r['description']); ?></td>
                        <td>
                            <?php if (strtolower($r['name']) !== 'superadmin'): ?>
                                <?php if (hasPermission('roles','edit')): ?>
                                    <a href="roles.php?action=edit&amp;id=<?php echo (int)$r['id']; ?>" class="btn btn-sm btn-primary">Editar</a>
                                <?php endif; ?>
                                <?php if (hasPermission('roles','delete')): ?>
                                    <a href="roles.php?action=delete&amp;id=<?php echo (int)$r['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar este rol?');">Eliminar</a>
                                <?php endif; ?>
                            <?php else: ?>
                                <em>--</em>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>
<?php
renderFooter();