<?php
/*
 * Gestión de usuarios: listado, creación, edición y eliminación.
 * Permite asignar un rol a cada usuario.
 */

require_once __DIR__ . '/includes/functions.php';
requireLogin();
// Permiso para ver usuarios
requirePermission('users', 'view');

$pdo = getPDO();
if (!$pdo) {
    echo '<div class="container mt-5"><div class="alert alert-danger">No se pudo conectar a la base de datos.</div></div>';
    exit;
}
ensureRoleTables($pdo);

$action = $_GET['action'] ?? 'list';
$userId = isset($_GET['id']) ? (int)$_GET['id'] : 0;

// Manejar eliminación
if ($action === 'delete' && $userId > 0) {
    requirePermission('users', 'delete');
    deleteUser($pdo, $userId);
    redirect('users.php');
}

// Procesar formulario de alta / edición
$errorUser = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $username   = trim($_POST['username'] ?? '');
    $password   = $_POST['password'] ?? '';
    $password2  = $_POST['password2'] ?? '';
    $roleId     = isset($_POST['role_id']) ? (int)$_POST['role_id'] : 0;
    if ($username === '' || $roleId <= 0) {
        $errorUser = 'El nombre de usuario y el rol son obligatorios.';
    } else {
        if ($action === 'create') {
            requirePermission('users', 'create');
            if ($password === '' || $password2 === '') {
                $errorUser = 'La contraseña es obligatoria.';
            } elseif ($password !== $password2) {
                $errorUser = 'Las contraseñas no coinciden.';
            } else {
                // Verificar que el usuario no exista
                $stmt = $pdo->prepare('SELECT id FROM users WHERE username = :u');
                $stmt->execute([':u' => $username]);
                if ($stmt->fetch()) {
                    $errorUser = 'El nombre de usuario ya existe.';
                } else {
                    createUser($pdo, $username, $password, $roleId);
                    redirect('users.php');
                }
            }
        } elseif ($action === 'edit' && $userId > 0) {
            requirePermission('users', 'edit');
            // No permitir que un usuario sin privilegios modifique SuperAdmin ni eliminarse a sí mismo
            $userRow = getUserById($pdo, $userId);
            if (!$userRow) {
                $errorUser = 'Usuario no encontrado.';
            } else {
                // Si se van a cambiar contraseñas
                $passToSet = null;
                if ($password !== '' || $password2 !== '') {
                    if ($password !== $password2) {
                        $errorUser = 'Las contraseñas no coinciden.';
                    } else {
                        $passToSet = $password;
                    }
                }
                if ($errorUser === '') {
                    updateUser($pdo, $userId, $username, $passToSet, $roleId);
                    redirect('users.php');
                }
            }
        }
    }
}

require_once __DIR__ . '/layout.php';
renderHeader('Usuarios', 'users');

?>
<div class="container-fluid">
    <?php if ($action === 'create' || ($action === 'edit' && $userId > 0)): ?>
        <?php
        $editing = ($action === 'edit');
        $uData = ['username' => '', 'role_id' => 0];
        if ($editing) {
            $userRow = getUserById($pdo, $userId);
            if (!$userRow) {
                echo '<div class="alert alert-danger">Usuario no encontrado.</div>';
            } else {
                $uData['username'] = $userRow['username'];
                $uData['role_id']  = $userRow['role_id'] ?? 0;
            }
        }
        // Obtener lista de roles
        $rolesList = getRoles($pdo);
        ?>
        <h1 class="mb-4"><?php echo $editing ? 'Editar usuario' : 'Crear usuario'; ?></h1>
        <?php if ($errorUser): ?>
            <div class="alert alert-danger"><?php echo escape($errorUser); ?></div>
        <?php endif; ?>
        <form method="post" action="">
            <div class="mb-3">
                <label for="username" class="form-label">Usuario</label>
                <input type="text" class="form-control" id="username" name="username" required value="<?php echo escape($uData['username']); ?>">
            </div>
            <div class="row">
                <div class="col-md-6 mb-3">
                    <label for="password" class="form-label">Contraseña <?php echo $editing ? '(dejar en blanco para mantener)' : ''; ?></label>
                    <input type="password" class="form-control" id="password" name="password">
                </div>
                <div class="col-md-6 mb-3">
                    <label for="password2" class="form-label">Confirmar contraseña</label>
                    <input type="password" class="form-control" id="password2" name="password2">
                </div>
            </div>
            <div class="mb-3">
                <label for="role_id" class="form-label">Rol</label>
                <select class="form-select" id="role_id" name="role_id" required>
                    <option value="">Seleccione un rol</option>
                    <?php foreach ($rolesList as $r): ?>
                        <option value="<?php echo (int)$r['id']; ?>"<?php echo ($uData['role_id'] == $r['id']) ? ' selected' : ''; ?>><?php echo escape($r['name']); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            <button type="submit" class="btn btn-primary"><?php echo $editing ? 'Actualizar' : 'Crear'; ?></button>
            <a href="users.php" class="btn btn-secondary">Cancelar</a>
        </form>
    <?php else: ?>
        <h1 class="mb-4">Usuarios</h1>
        <?php if (hasPermission('users','create')): ?>
            <a href="users.php?action=create" class="btn btn-success mb-3">Nuevo usuario</a>
        <?php endif; ?>
        <div class="table-responsive">
            <table class="table table-striped table-bordered">
                <thead>
                <tr>
                    <th>ID</th>
                    <th>Usuario</th>
                    <th>Rol</th>
                    <th>Acciones</th>
                </tr>
                </thead>
                <tbody>
                <?php foreach (getUsers($pdo) as $usr): ?>
                    <tr>
                        <td><?php echo (int)$usr['id']; ?></td>
                        <td><?php echo escape($usr['username']); ?></td>
                        <td><?php echo escape($usr['role_name'] ?? ($usr['role'] ?? '')); ?></td>
                        <td>
                            <?php if (hasPermission('users','edit')): ?>
                                <a href="users.php?action=edit&amp;id=<?php echo (int)$usr['id']; ?>" class="btn btn-sm btn-primary">Editar</a>
                            <?php endif; ?>
                            <?php if (hasPermission('users','delete')): ?>
                                <?php if ($_SESSION['user_id'] != $usr['id']): ?>
                                    <a href="users.php?action=delete&amp;id=<?php echo (int)$usr['id']; ?>" class="btn btn-sm btn-danger" onclick="return confirm('¿Está seguro de eliminar este usuario?');">Eliminar</a>
                                <?php else: ?>
                                    <em>--</em>
                                <?php endif; ?>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
        </div>
    <?php endif; ?>
</div>
<?php
renderFooter();