<?php
/*
 * Punto de Venta (Hito 5).
 *
 * Esta página implementa una interfaz responsiva para realizar ventas de forma fluida. Permite
 * buscar productos (lectura básica desde la BD de PrestaShop), agregarlos a un carrito local,
 * modificar cantidades, quitar productos, seleccionar un medio de pago y calcular el total
 * con el descuento correspondiente. Aún no se crean pedidos reales ni se descuenta el stock.
 */

require_once __DIR__ . '/includes/functions.php';
requireLogin();
requirePermission('pos', 'view');

$pdo = getPDO();
// Asegurar que existan los medios de pago y cargarlos
ensurePaymentTables($pdo);

// Inicializar carrito en sesión si no existe
if (!isset($_SESSION['cart']) || !is_array($_SESSION['cart'])) {
    $_SESSION['cart'] = [];
}

// Obtener lista de productos según búsqueda
$searchTerm = trim($_GET['search'] ?? '');
$products   = getProducts($searchTerm);

// Manejar acciones del carrito
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    // Añadir producto al carrito
    if ($action === 'add') {
        $pid = (int)($_POST['product_id'] ?? 0);
        if ($pid > 0) {
            $prod = getProductById($pid);
            if ($prod) {
                // Si ya existe en el carrito, incrementar cantidad
                if (isset($_SESSION['cart'][$pid])) {
                    $_SESSION['cart'][$pid]['qty'] += 1;
                } else {
                    $_SESSION['cart'][$pid] = [
                        'id'    => $prod['id'],
                        'name'  => $prod['name'],
                        'price' => $prod['price'],
                        'qty'   => 1,
                    ];
                }
            }
        }
        // Mantener término de búsqueda al redirigir
        $redirect = 'pos.php';
        if ($searchTerm !== '') {
            $redirect .= '?search=' . urlencode($searchTerm);
        }
        redirect($redirect);
    }
    // Actualizar cantidades y método de pago
    if ($action === 'update') {
        // Actualizar cantidades
        if (isset($_POST['quantities']) && is_array($_POST['quantities'])) {
            foreach ($_POST['quantities'] as $pid => $qty) {
                $pid = (int)$pid;
                $qty = (int)$qty;
                if (isset($_SESSION['cart'][$pid])) {
                    if ($qty > 0) {
                        $_SESSION['cart'][$pid]['qty'] = $qty;
                    } else {
                        unset($_SESSION['cart'][$pid]);
                    }
                }
            }
        }
        // Actualizar método de pago seleccionado
        $selectedPayment = isset($_POST['payment_id']) ? (int)$_POST['payment_id'] : 0;
        $_SESSION['selected_payment_id'] = $selectedPayment;
        // Redirigir para evitar reenvío
        $redirect = 'pos.php';
        if ($searchTerm !== '') {
            $redirect .= '?search=' . urlencode($searchTerm);
        }
        redirect($redirect);
    }
    // Quitar un producto
    if ($action === 'remove') {
        $pid = (int)($_POST['product_id'] ?? 0);
        if ($pid > 0 && isset($_SESSION['cart'][$pid])) {
            unset($_SESSION['cart'][$pid]);
        }
        $redirect = 'pos.php';
        if ($searchTerm !== '') {
            $redirect .= '?search=' . urlencode($searchTerm);
        }
        redirect($redirect);
    }
    // Vaciar carrito
    if ($action === 'clear') {
        $_SESSION['cart'] = [];
        $_SESSION['selected_payment_id'] = 0;
        $redirect = 'pos.php';
        if ($searchTerm !== '') {
            $redirect .= '?search=' . urlencode($searchTerm);
        }
        redirect($redirect);
    }
    // Finalizar venta (simulado)
    if ($action === 'finish') {
        // Para el hito 5, solo se muestra un mensaje informativo. La creación real de pedidos se implementará en el hito 7.
        $_SESSION['cart'] = [];
        $_SESSION['selected_payment_id'] = 0;
        $redirect = 'pos.php';
        $redirect .= '?msg=venta';
        redirect($redirect);
    }
}

// Obtener carrito y método de pago seleccionado
$cart = $_SESSION['cart'];
$selectedPaymentId = $_SESSION['selected_payment_id'] ?? 0;

// Calcular totales
$subtotal = 0;
foreach ($cart as $item) {
    $subtotal += $item['price'] * $item['qty'];
}
// Determinar descuento
$discountPercent = 0;
$selectedPayment = null;
if ($selectedPaymentId) {
    $pm = getPaymentMethodById($pdo, $selectedPaymentId);
    if ($pm && $pm['active']) {
        $discountPercent = (float)$pm['discount_percent'];
        $selectedPayment = $pm;
    }
}
$discountAmount = $subtotal * $discountPercent / 100;
$total = $subtotal - $discountAmount;

require_once __DIR__ . '/layout.php';
renderHeader('Punto de Venta', 'pos');
?>
<div class="container-fluid">
    <h1 class="mb-4">Punto de Venta</h1>
    <?php if (isset($_GET['msg']) && $_GET['msg'] === 'venta'): ?>
        <div class="alert alert-success">Venta registrada localmente (no se ha creado un pedido real). El flujo completo se implementará en el hito 7.</div>
    <?php endif; ?>
    <div class="row">
        <!-- Listado de productos -->
        <div class="col-lg-8 mb-4">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Productos</h5>
                    <form class="d-flex" method="get" action="pos.php">
                        <input type="text" name="search" class="form-control me-2" placeholder="Buscar..." value="<?php echo escape($searchTerm); ?>">
                        <button class="btn btn-outline-primary" type="submit">Buscar</button>
                    </form>
                </div>
                <div class="card-body p-2" style="min-height:400px; overflow-y:auto;">
                    <?php if (empty($products)): ?>
                        <p class="text-muted">No se encontraron productos.</p>
                    <?php else: ?>
                        <div class="row g-2">
                            <?php foreach ($products as $prod): ?>
                                <div class="col-6 col-md-4 col-lg-3">
                                    <div class="card h-100">
                                        <div class="card-body p-2 d-flex flex-column justify-content-between">
                                            <div>
                                                <h6 class="card-title mb-1" title="<?php echo escape($prod['name']); ?>"><?php echo escape($prod['name']); ?></h6>
                                                <p class="card-text mb-1"><small><?php echo escape($prod['reference']); ?></small></p>
                                            </div>
                                            <div>
                                                <strong><?php echo number_format($prod['price'], 2); ?> €</strong>
                                                <form method="post" action="" class="mt-2">
                                                    <input type="hidden" name="action" value="add">
                                                    <input type="hidden" name="product_id" value="<?php echo (int)$prod['id']; ?>">
                                                    <button type="submit" class="btn btn-sm btn-primary w-100"><i class="fas fa-plus"></i> Agregar</button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <!-- Carrito -->
        <div class="col-lg-4 mb-4">
            <div class="card">
                <div class="card-header"><h5 class="mb-0">Carrito</h5></div>
                <div class="card-body">
                    <?php if (empty($cart)): ?>
                        <p class="text-muted">No hay productos en el carrito.</p>
                    <?php else: ?>
                        <form method="post" action="">
                            <input type="hidden" name="action" value="update">
                            <div class="table-responsive" style="max-height:300px;">
                                <table class="table table-sm align-middle">
                                    <thead>
                                    <tr>
                                        <th>Producto</th>
                                        <th class="text-center" style="width:70px;">Cant.</th>
                                        <th class="text-end">Precio</th>
                                        <th class="text-end">Total</th>
                                        <th></th>
                                    </tr>
                                    </thead>
                                    <tbody>
                                    <?php foreach ($cart as $id => $item): ?>
                                        <tr>
                                            <td><?php echo escape($item['name']); ?></td>
                                            <td class="text-center">
                                                <input type="number" min="1" class="form-control form-control-sm" name="quantities[<?php echo (int)$id; ?>]" value="<?php echo (int)$item['qty']; ?>">
                                            </td>
                                            <td class="text-end"><?php echo number_format($item['price'], 2); ?> €</td>
                                            <td class="text-end"><?php echo number_format($item['price'] * $item['qty'], 2); ?> €</td>
                                            <td>
                                                <form method="post" action="" onsubmit="return confirm('¿Quitar producto?');">
                                                    <input type="hidden" name="action" value="remove">
                                                    <input type="hidden" name="product_id" value="<?php echo (int)$id; ?>">
                                                    <button type="submit" class="btn btn-sm btn-link text-danger"><i class="fas fa-trash"></i></button>
                                                </form>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                            <!-- Selección de medio de pago -->
                            <div class="mb-2">
                                <label for="payment_id" class="form-label">Medio de pago</label>
                                <select class="form-select" id="payment_id" name="payment_id">
                                    <option value="0">-- Seleccionar --</option>
                                    <?php foreach (getPaymentMethods($pdo) as $pm): ?>
                                        <?php if ($pm['active']): ?>
                                            <option value="<?php echo (int)$pm['id']; ?>" <?php echo ($selectedPaymentId == $pm['id']) ? 'selected' : ''; ?>><?php echo escape($pm['name']); ?> (<?php echo number_format($pm['discount_percent'], 2); ?>% de desc.)</option>
                                        <?php endif; ?>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="mb-2 d-flex justify-content-between">
                                <div>Subtotal:</div>
                                <div><?php echo number_format($subtotal, 2); ?> €</div>
                            </div>
                            <div class="mb-2 d-flex justify-content-between">
                                <div>Descuento (<?php echo number_format($discountPercent, 2); ?>%):</div>
                                <div>- <?php echo number_format($discountAmount, 2); ?> €</div>
                            </div>
                            <div class="mb-3 d-flex justify-content-between fs-5 fw-bold">
                                <div>Total:</div>
                                <div><?php echo number_format($total, 2); ?> €</div>
                            </div>
                            <div class="d-flex justify-content-between">
                                <button type="submit" class="btn btn-primary">Actualizar</button>
                                <form method="post" action="">
                                    <input type="hidden" name="action" value="clear">
                                    <button type="submit" class="btn btn-secondary" onclick="return confirm('¿Vaciar carrito?');">Vaciar</button>
                                </form>
                            </div>
                        </form>
                        <form method="post" action="" class="mt-3">
                            <input type="hidden" name="action" value="finish">
                            <button type="submit" class="btn btn-success w-100" <?php echo empty($cart) ? 'disabled' : ''; ?>>Finalizar venta (simulado)</button>
                        </form>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
</div>
<?php renderFooter(); ?>