<?php
/**
 * Funciones auxiliares para el sistema PDV
 */

// Iniciar la sesión sólo si aún no está iniciada
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * Configura el manejo de errores y excepciones.
 *
 * Si la opción `app.debug` en la configuración es true, se mostrarán los errores
 * y se guardarán en un archivo de registro en pos/logs/error.log. De lo contrario
 * se ocultarán los errores al usuario final y solo se registrarán.
 */
function setupErrorHandling(): void
{
    // Determinar si está activo el modo debug
    $config = loadConfig();
    $debug  = false;
    if ($config && isset($config['app']['debug'])) {
        $debug = (bool)$config['app']['debug'];
    }
    // Ajustar directivas de PHP según debug
    if ($debug) {
        ini_set('display_errors', '1');
        ini_set('display_startup_errors', '1');
        error_reporting(E_ALL);
    } else {
        ini_set('display_errors', '0');
        ini_set('display_startup_errors', '0');
    }
    // Asegurarse de que exista el directorio de logs
    $logsDir = __DIR__ . '/../logs';
    if (!is_dir($logsDir)) {
        @mkdir($logsDir, 0775, true);
    }
    // Manejador de errores personalizado
    set_error_handler(function ($severity, $message, $file, $line) use ($debug) {
        // Crear mensaje
        $msg = "Error [$severity] $message in $file on line $line";
        // Registrar en archivo
        logError($msg);
        // Si está en modo debug, mostrar alerta
        if ($debug) {
            // Sanitizar el mensaje antes de mostrar
            echo '<div class="alert alert-danger m-3">' . escape($msg) . '</div>';
        }
        // Permitir que PHP continúe con su manejador por defecto
        return false;
    });
    // Manejador de excepciones personalizado
    set_exception_handler(function ($e) use ($debug) {
        $msg = 'Exception: ' . $e->getMessage() . ' in ' . $e->getFile() . ':' . $e->getLine();
        logError($msg);
        if ($debug) {
            echo '<div class="alert alert-danger m-3">' . escape($msg) . '</div>';
        } else {
            // En entorno no debug, mostrar mensaje genérico al usuario
            echo '<div class="alert alert-danger m-3">Ha ocurrido un error inesperado. Por favor, contacte al administrador.</div>';
        }
    });
}

/**
 * Registra un mensaje en el archivo de log de errores.
 *
 * @param string $message
 */
function logError(string $message): void
{
    $dir = __DIR__ . '/../logs';
    // Crear el directorio si no existe
    if (!is_dir($dir)) {
        @mkdir($dir, 0775, true);
    }
    $file = $dir . '/error.log';
    $date = date('Y-m-d H:i:s');
    @file_put_contents($file, "[$date] $message\n", FILE_APPEND);
}

// Configurar el manejo de errores tan pronto como se cargan las funciones
setupErrorHandling();

/**
 * Carga la configuración desde el archivo config.php.
 *
 * @return array|null Devuelve la configuración o null si no existe.
 */
function loadConfig(): ?array
{
    $configFile = __DIR__ . '/config.php';
    if (file_exists($configFile)) {
        return require $configFile;
    }
    return null;
}

/**
 * Devuelve una conexión PDO a la base de datos del PDV.
 *
 * @return PDO|null
 */
function getPDO(): ?PDO
{
    static $pdo;
    if ($pdo instanceof PDO) {
        return $pdo;
    }
    $config = loadConfig();
    if (!$config || empty($config['pdv'])) {
        return null;
    }
    $db = $config['pdv'];
    $dsn = sprintf('mysql:host=%s;dbname=%s;charset=%s', $db['host'], $db['database'], $db['charset'] ?? 'utf8mb4');
    try {
        $pdo = new PDO($dsn, $db['username'], $db['password'], [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ]);
        return $pdo;
    } catch (PDOException $e) {
        // No exponer detalles sensibles
        return null;
    }
}

/**
 * Escapa una cadena para mostrarla en HTML.
 *
 * @param string|null $str
 * @return string
 */
function escape(?string $str): string
{
    return htmlspecialchars($str ?? '', ENT_QUOTES, 'UTF-8');
}

/**
 * Redirige a la URL indicada y detiene la ejecución.
 *
 * @param string $url
 */
function redirect(string $url): void
{
    header('Location: ' . $url);
    exit;
}

/**
 * Genera una cadena aleatoria de la longitud especificada.
 *
 * @param int $length
 * @return string
 */
function randomString(int $length = 32): string
{
    return bin2hex(random_bytes($length / 2));
}

/**
 * Devuelve la URL base de la aplicación.
 *
 * Si se ha configurado `app.base_url` se usará ese valor. En caso contrario,
 * se construirá a partir de la URL actual sin la parte de la ruta del script.
 *
 * @param string $path Ruta opcional para añadir a la base.
 * @return string
 */
function baseUrl(string $path = ''): string
{
    $config = loadConfig();
    $base = '';
    if ($config && !empty($config['app']['base_url'])) {
        $base = rtrim($config['app']['base_url'], '/');
    } else {
        // Construir base URL a partir de la petición
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        // Directorio del script actual
        $scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/');
        $base = $protocol . '://' . $host . $scriptDir;
    }
    if ($path) {
        return $base . '/' . ltrim($path, '/');
    }
    return $base;
}

/**
 * Comprueba si el usuario está logueado.
 *
 * @return bool
 */
function isLoggedIn(): bool
{
    return isset($_SESSION['user_id']);
}

/**
 * Obliga a que el usuario esté logueado. Si no lo está, redirige a login.php.
 */
function requireLogin(): void
{
    if (!isLoggedIn()) {
        redirect('login.php');
    }
}

/**
 * Devuelve la información del usuario actualmente logueado.
 *
 * @return array|null
 */
function currentUser(): ?array
{
    if (!isLoggedIn()) {
        return null;
    }
    $pdo = getPDO();
    if (!$pdo) {
        return null;
    }
    $stmt = $pdo->prepare('SELECT * FROM users WHERE id = :id');
    $stmt->execute([':id' => $_SESSION['user_id']]);
    return $stmt->fetch() ?: null;
}

/**
 * Crea las tablas necesarias para el PDV en la base de datos.
 *
 * @param PDO $pdo
 */
function createTables(PDO $pdo): void
{
    // Tabla de usuarios
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS users (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    username VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    password VARCHAR(255) NOT NULL,\n" .
        "    role VARCHAR(50) NOT NULL DEFAULT 'superadmin',\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );

    // Tabla de configuraciones (clave/valor)
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS settings (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    value TEXT NOT NULL\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
}

/**
 * Crea las tablas necesarias para roles y permisos si no existen y ajusta la tabla de usuarios.
 *
 * @param PDO $pdo
 */
function ensureRoleTables(PDO $pdo): void
{
    // Tabla de roles
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS roles (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    description VARCHAR(255) NULL,\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );

    // Tabla de permisos por rol y módulo
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS role_permissions (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    role_id INT NOT NULL,\n" .
        "    module VARCHAR(100) NOT NULL,\n" .
        "    can_view TINYINT(1) DEFAULT 0,\n" .
        "    can_create TINYINT(1) DEFAULT 0,\n" .
        "    can_edit TINYINT(1) DEFAULT 0,\n" .
        "    can_delete TINYINT(1) DEFAULT 0,\n" .
        "    UNIQUE KEY role_module (role_id, module),\n" .
        "    CONSTRAINT fk_rp_role FOREIGN KEY (role_id) REFERENCES roles(id) ON DELETE CASCADE\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );

    // Añadir columna role_id a la tabla de usuarios si no existe
    $cols = $pdo->query("SHOW COLUMNS FROM users LIKE 'role_id'")->fetch();
    if (!$cols) {
        $pdo->exec("ALTER TABLE users ADD COLUMN role_id INT NULL AFTER role");
    }

    // Asegurar el rol SuperAdmin
    $stmt = $pdo->prepare('SELECT id FROM roles WHERE name = :name');
    $stmt->execute([':name' => 'SuperAdmin']);
    $superRoleId = $stmt->fetchColumn();
    if (!$superRoleId) {
        // Crear rol SuperAdmin con descripción
        $pdo->prepare('INSERT INTO roles (name, description) VALUES (:name, :desc)')->execute([
            ':name' => 'SuperAdmin',
            ':desc' => 'Rol con permisos completos en el sistema',
        ]);
        $superRoleId = (int)$pdo->lastInsertId();
    }

    // Asignar rol SuperAdmin a usuarios con campo role='superadmin' (valor original)
    $stmt = $pdo->prepare('UPDATE users SET role_id = :role_id WHERE LOWER(role) = :super AND (role_id IS NULL OR role_id = 0)');
    $stmt->execute([
        ':role_id' => $superRoleId,
        ':super'   => 'superadmin',
    ]);

    // Asignar permisos completos al rol SuperAdmin si no existen
    $modules = getModules();
    foreach ($modules as $module) {
        $stmt = $pdo->prepare('SELECT id FROM role_permissions WHERE role_id = :role_id AND module = :module');
        $stmt->execute([
            ':role_id' => $superRoleId,
            ':module'  => $module,
        ]);
        $exists = $stmt->fetch();
        if (!$exists) {
            $pdo->prepare(
                'INSERT INTO role_permissions (role_id, module, can_view, can_create, can_edit, can_delete) VALUES (:role_id, :module, 1, 1, 1, 1)'
            )->execute([
                ':role_id' => $superRoleId,
                ':module'  => $module,
            ]);
        }
    }
}

/**
 * Devuelve la lista de módulos disponibles para permisos.
 *
 * @return string[]
 */
function getModules(): array
{
    return [
        'dashboard',
        'users',
        'roles',
        'boxes',
        'settings',
        'pos',
        'stats',
    ];
}

/**
 * Obtiene todos los roles disponibles.
 *
 * @param PDO $pdo
 * @return array
 */
function getRoles(PDO $pdo): array
{
    $stmt = $pdo->query('SELECT * FROM roles ORDER BY id ASC');
    return $stmt->fetchAll();
}

/**
 * Obtiene la información de un rol por su ID.
 *
 * @param PDO $pdo
 * @param int $roleId
 * @return array|null
 */
function getRoleById(PDO $pdo, int $roleId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM roles WHERE id = :id');
    $stmt->execute([':id' => $roleId]);
    $role = $stmt->fetch();
    return $role ?: null;
}

/**
 * Crea un rol con sus permisos y devuelve su ID.
 *
 * @param PDO $pdo
 * @param string $name
 * @param string $description
 * @param array $permissions Formato: ['module' => ['view' => bool, 'create' => bool, ...], ...]
 * @return int
 */
function createRole(PDO $pdo, string $name, string $description, array $permissions): int
{
    // Insertar rol
    $stmt = $pdo->prepare('INSERT INTO roles (name, description) VALUES (:name, :description)');
    $stmt->execute([
        ':name'        => $name,
        ':description' => $description,
    ]);
    $roleId = (int)$pdo->lastInsertId();
    // Guardar permisos
    saveRolePermissions($pdo, $roleId, $permissions);
    return $roleId;
}

/**
 * Actualiza un rol y sus permisos.
 *
 * @param PDO $pdo
 * @param int $id
 * @param string $name
 * @param string $description
 * @param array $permissions
 */
function updateRole(PDO $pdo, int $id, string $name, string $description, array $permissions): void
{
    $stmt = $pdo->prepare('UPDATE roles SET name = :name, description = :description WHERE id = :id');
    $stmt->execute([
        ':name'        => $name,
        ':description' => $description,
        ':id'          => $id,
    ]);
    // Actualizar permisos
    saveRolePermissions($pdo, $id, $permissions);
}

/**
 * Elimina un rol y sus permisos.
 * No permite eliminar el rol SuperAdmin.
 *
 * @param PDO $pdo
 * @param int $id
 */
function deleteRole(PDO $pdo, int $id): void
{
    // Comprobar si es SuperAdmin
    $role = getRoleById($pdo, $id);
    if (!$role) {
        return;
    }
    if (strtolower($role['name']) === 'superadmin') {
        return; // No eliminar
    }
    // Eliminar usuarios que tengan este rol: reasignar a null o superadmin
    $stmt = $pdo->prepare('UPDATE users SET role_id = NULL WHERE role_id = :id');
    $stmt->execute([':id' => $id]);
    // Eliminar permisos y rol
    $pdo->prepare('DELETE FROM role_permissions WHERE role_id = :id')->execute([':id' => $id]);
    $pdo->prepare('DELETE FROM roles WHERE id = :id')->execute([':id' => $id]);
}

/**
 * Obtiene los permisos de un rol.
 *
 * @param PDO $pdo
 * @param int $roleId
 * @return array Formato: [module => ['view' => bool, 'create' => bool, 'edit' => bool, 'delete' => bool], ...]
 */
function getRolePermissions(PDO $pdo, int $roleId): array
{
    $modules = getModules();
    $permissions = [];
    foreach ($modules as $module) {
        $permissions[$module] = [
            'view'   => false,
            'create' => false,
            'edit'   => false,
            'delete' => false,
        ];
    }
    $stmt = $pdo->prepare('SELECT module, can_view, can_create, can_edit, can_delete FROM role_permissions WHERE role_id = :role_id');
    $stmt->execute([':role_id' => $roleId]);
    foreach ($stmt as $row) {
        $module = $row['module'];
        $permissions[$module] = [
            'view'   => (bool)$row['can_view'],
            'create' => (bool)$row['can_create'],
            'edit'   => (bool)$row['can_edit'],
            'delete' => (bool)$row['can_delete'],
        ];
    }
    return $permissions;
}

/**
 * Guarda los permisos de un rol.
 *
 * @param PDO $pdo
 * @param int $roleId
 * @param array $permissions
 */
function saveRolePermissions(PDO $pdo, int $roleId, array $permissions): void
{
    // Eliminar permisos existentes
    $stmtDelete = $pdo->prepare('DELETE FROM role_permissions WHERE role_id = :role_id');
    $stmtDelete->execute([':role_id' => $roleId]);
    // Insertar nuevos permisos
    $stmt = $pdo->prepare(
        'INSERT INTO role_permissions (role_id, module, can_view, can_create, can_edit, can_delete) VALUES (:role_id, :module, :view, :create, :edit, :delete)'
    );
    foreach ($permissions as $module => $acts) {
        $stmt->execute([
            ':role_id' => $roleId,
            ':module'  => $module,
            ':view'    => empty($acts['view'])   ? 0 : 1,
            ':create'  => empty($acts['create']) ? 0 : 1,
            ':edit'    => empty($acts['edit'])   ? 0 : 1,
            ':delete'  => empty($acts['delete']) ? 0 : 1,
        ]);
    }
}

/**
 * Obtiene todos los usuarios.
 *
 * @param PDO $pdo
 * @return array
 */
function getUsers(PDO $pdo): array
{
    $stmt = $pdo->query('SELECT users.*, roles.name AS role_name FROM users LEFT JOIN roles ON users.role_id = roles.id ORDER BY users.id ASC');
    return $stmt->fetchAll();
}

/**
 * Obtiene la información de un usuario por su ID.
 *
 * @param PDO $pdo
 * @param int $id
 * @return array|null
 */
function getUserById(PDO $pdo, int $id): ?array
{
    $stmt = $pdo->prepare('SELECT users.*, roles.name AS role_name FROM users LEFT JOIN roles ON users.role_id = roles.id WHERE users.id = :id');
    $stmt->execute([':id' => $id]);
    $user = $stmt->fetch();
    return $user ?: null;
}

/**
 * Crea un usuario y lo asigna a un rol.
 *
 * @param PDO $pdo
 * @param string $username
 * @param string $password
 * @param int $roleId
 * @return int
 */
function createUser(PDO $pdo, string $username, string $password, int $roleId): int
{
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare('INSERT INTO users (username, password, role_id) VALUES (:username, :password, :role_id)');
    $stmt->execute([
        ':username' => $username,
        ':password' => $hash,
        ':role_id'  => $roleId,
    ]);
    return (int)$pdo->lastInsertId();
}

/**
 * Actualiza un usuario y su rol. Si la contraseña es null se mantiene la actual.
 *
 * @param PDO $pdo
 * @param int $id
 * @param string $username
 * @param string|null $password
 * @param int $roleId
 */
function updateUser(PDO $pdo, int $id, string $username, ?string $password, int $roleId): void
{
    if ($password === null || $password === '') {
        $stmt = $pdo->prepare('UPDATE users SET username = :username, role_id = :role_id WHERE id = :id');
        $stmt->execute([
            ':username' => $username,
            ':role_id'  => $roleId,
            ':id'       => $id,
        ]);
    } else {
        $hash = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare('UPDATE users SET username = :username, password = :password, role_id = :role_id WHERE id = :id');
        $stmt->execute([
            ':username' => $username,
            ':password' => $hash,
            ':role_id'  => $roleId,
            ':id'       => $id,
        ]);
    }
}

/**
 * Elimina un usuario. No permite eliminar tu propia cuenta.
 *
 * @param PDO $pdo
 * @param int $id
 */
function deleteUser(PDO $pdo, int $id): void
{
    if (!isLoggedIn() || $_SESSION['user_id'] == $id) {
        return;
    }
    $stmt = $pdo->prepare('DELETE FROM users WHERE id = :id');
    $stmt->execute([':id' => $id]);
}

/**
 * Comprueba si el usuario actual tiene permiso sobre un módulo y acción.
 *
 * @param string $module
 * @param string $action view|create|edit|delete
 * @return bool
 */
function hasPermission(string $module, string $action = 'view'): bool
{
    $user = currentUser();
    if (!$user) {
        return false;
    }
    // Los superadministradores siempre tienen permiso
    if (isset($user['role']) && strtolower($user['role']) === 'superadmin') {
        return true;
    }
    $pdo = getPDO();
    if (!$pdo) {
        return false;
    }
    // Asegurar tablas de roles
    ensureRoleTables($pdo);
    // Obtener role_id
    $roleId = $user['role_id'] ?? null;
    if (!$roleId) {
        // Si no tiene role_id, buscar por el nombre del rol
        $roleName = $user['role'] ?? '';
        if ($roleName) {
            $stmt = $pdo->prepare('SELECT id FROM roles WHERE name = :name');
            $stmt->execute([':name' => $roleName]);
            $roleId = $stmt->fetchColumn();
        }
    }
    if (!$roleId) {
        return false;
    }
    // Consultar permisos
    $stmt = $pdo->prepare('SELECT can_view, can_create, can_edit, can_delete FROM role_permissions WHERE role_id = :role_id AND module = :module');
    $stmt->execute([
        ':role_id' => $roleId,
        ':module'  => $module,
    ]);
    $row = $stmt->fetch();
    if (!$row) {
        return false;
    }
    switch ($action) {
        case 'view':
            return (bool)$row['can_view'];
        case 'create':
            return (bool)$row['can_create'];
        case 'edit':
            return (bool)$row['can_edit'];
        case 'delete':
            return (bool)$row['can_delete'];
        default:
            return false;
    }
}

/**
 * Requiere que el usuario tenga permiso para acceder al módulo y acción especificados.
 * Si no lo tiene, redirige a la página principal o muestra un mensaje.
 *
 * @param string $module
 * @param string $action
 */
function requirePermission(string $module, string $action = 'view'): void
{
    if (!hasPermission($module, $action)) {
        // Mostrar mensaje o redirigir
        echo '<div class="container mt-5"><div class="alert alert-danger">No tiene permiso para acceder a esta sección.</div></div>';
        exit;
    }
}

/**
 * Asegura la existencia de las tablas de cajas y sesiones de caja.
 *
 * @param PDO $pdo
 */
function ensureBoxTables(PDO $pdo): void
{
    // Tabla de cajas
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS boxes (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    description VARCHAR(255) NULL,\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
    // Sesiones de caja
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS cash_sessions (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    box_id INT NOT NULL,\n" .
        "    user_id INT NOT NULL,\n" .
        "    opened_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n" .
        "    initial_amount DECIMAL(15,2) NOT NULL,\n" .
        "    closed_at TIMESTAMP NULL DEFAULT NULL,\n" .
        "    closed_by INT NULL,\n" .
        "    final_amount DECIMAL(15,2) NULL,\n" .
        "    difference DECIMAL(15,2) NULL,\n" .
        "    CONSTRAINT fk_sessions_box FOREIGN KEY (box_id) REFERENCES boxes(id) ON DELETE CASCADE,\n" .
        "    CONSTRAINT fk_sessions_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
    // Movimientos de caja
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS cash_movements (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    session_id INT NOT NULL,\n" .
        "    user_id INT NOT NULL,\n" .
        "    movement_type ENUM('ingreso','retiro') NOT NULL,\n" .
        "    amount DECIMAL(15,2) NOT NULL,\n" .
        "    description VARCHAR(255) NULL,\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,\n" .
        "    CONSTRAINT fk_movements_session FOREIGN KEY (session_id) REFERENCES cash_sessions(id) ON DELETE CASCADE,\n" .
        "    CONSTRAINT fk_movements_user FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
}

/**
 * Asegura la existencia de la tabla de medios de pago.
 *
 * Cada medio de pago tiene un nombre, un estado de activación y un porcentaje de descuento.
 *
 * @param PDO $pdo
 */
function ensurePaymentTables(PDO $pdo): void
{
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS payment_methods (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    name VARCHAR(100) NOT NULL UNIQUE,\n" .
        "    module_name VARCHAR(100) DEFAULT '',\n" .
        "    active TINYINT(1) NOT NULL DEFAULT 1,\n" .
        "    discount_percent DECIMAL(5,2) NOT NULL DEFAULT 0,\n" .
        "    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
    // Si la columna module_name no existe en instalaciones antiguas, añadirla
    try {
        $pdo->exec("ALTER TABLE payment_methods ADD COLUMN module_name VARCHAR(100) DEFAULT ''");
    } catch (PDOException $e) {
        // Ignorar error si la columna ya existe
    }
}

/**
 * Asegura las tablas de ventas y sus detalles.
 *
 * Se crean las tablas `sales` y `sales_items` si no existen. La tabla `sales` almacena
 * la referencia a la venta de PDV y el ID de pedido generado en PrestaShop, así como
 * información de totales y método de pago. La tabla `sales_items` guarda el detalle
 * de cada producto vendido para futuras auditorías.
 *
 * @param PDO $pdo Conexión PDO al sistema PDV
 */
function ensureSalesTables(PDO $pdo): void
{
    // Tabla de ventas principales
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS sales (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    user_id INT NOT NULL,\n" .
        "    payment_method_id INT NOT NULL,\n" .
        "    order_id INT NOT NULL,\n" .
        "    subtotal DECIMAL(20,6) NOT NULL DEFAULT 0,\n" .
        "    discount DECIMAL(20,6) NOT NULL DEFAULT 0,\n" .
        "    total DECIMAL(20,6) NOT NULL DEFAULT 0,\n" .
        "    created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,\n" .
        "    FOREIGN KEY (payment_method_id) REFERENCES payment_methods(id) ON DELETE RESTRICT ON UPDATE CASCADE\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
    // Tabla de detalle de ventas
    $pdo->exec(
        "CREATE TABLE IF NOT EXISTS sales_items (\n" .
        "    id INT AUTO_INCREMENT PRIMARY KEY,\n" .
        "    sale_id INT NOT NULL,\n" .
        "    product_id INT NOT NULL,\n" .
        "    attribute_id INT NOT NULL DEFAULT 0,\n" .
        "    name VARCHAR(255) NOT NULL,\n" .
        "    reference VARCHAR(64) NOT NULL,\n" .
        "    quantity INT NOT NULL DEFAULT 0,\n" .
        "    unit_price DECIMAL(20,6) NOT NULL DEFAULT 0,\n" .
        "    FOREIGN KEY (sale_id) REFERENCES sales(id) ON DELETE CASCADE ON UPDATE CASCADE\n" .
        ") ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;"
    );
}

/**
 * Crea un pedido real en PrestaShop y registra la venta en la tabla `sales`.
 *
 * Este método utiliza el core de PrestaShop para generar un nuevo carrito y validar
 * el pedido mediante el módulo de pago configurado en el medio de pago seleccionado.
 * Se descuenta el stock automáticamente y se devuelve el ID del pedido creado.
 *
 * @param int   $userId        ID del usuario PDV que realiza la venta
 * @param array $cartItems     Lista de artículos en el carrito local. Cada elemento debe contener:
 *                             - product_id
 *                             - attribute_id
 *                             - qty
 *                             - name
 *                             - reference
 *                             - price
 * @param array $paymentMethod Registro del medio de pago (incluye `module_name`)
 * @param float $subtotal      Subtotal de la venta antes de descuentos
 * @param float $discount      Importe total descontado
 * @param float $total         Total de la venta (subtotal - descuento)
 * @return int                 Devuelve el ID del pedido de PrestaShop al finalizar
 * @throws Exception           Lanza una excepción si la integración con PrestaShop falla
 */
function createSaleAndOrder(int $userId, array $cartItems, array $paymentMethod, float $subtotal, float $discount, float $total)
{
    // Obtener configuración y ruta de PrestaShop
    $config = loadConfig();
    if (!$config || empty($config['prestashop'])) {
        throw new Exception('No se pudo cargar la configuración de PrestaShop');
    }
    $ps = $config['prestashop'];
    $psPath = $ps['path'] ?? '../';
    // Normalizar ruta (relativa a este script)
    $psRoot = realpath(__DIR__ . '/' . $psPath);
    if ($psRoot === false || !is_dir($psRoot)) {
        throw new Exception('Ruta de PrestaShop no encontrada: ' . $psPath);
    }
    // Incluir el núcleo de PrestaShop
    $configFile = $psRoot . '/config/config.inc.php';
    if (!file_exists($configFile)) {
        throw new Exception('No se encontró el archivo config.inc.php de PrestaShop en ' . $psRoot);
    }
    require_once $configFile;
    // Cargar init.php para inicializar el contexto de PrestaShop
    if (file_exists($psRoot . '/init.php')) {
        require_once $psRoot . '/init.php';
    }
    // Inicializar contexto
    $context = Context::getContext();
    // Seleccionar shop por defecto
    if (!isset($context->shop) || !$context->shop->id) {
        $context->shop = new Shop((int)Configuration::get('PS_SHOP_DEFAULT'));
    }
    // Crear nuevo carrito
    $cart = new Cart();
    // Seleccionar un cliente invitado o por defecto
    $customerId = 0;
    if ($context->customer && $context->customer->id) {
        $customerId = (int)$context->customer->id;
    }
    // Si no hay cliente en contexto, buscar el primer cliente activo
    if (!$customerId) {
        $row = Db::getInstance()->getRow('SELECT id_customer FROM ' . _DB_PREFIX_ . 'customer WHERE active = 1 ORDER BY id_customer ASC');
        if ($row) {
            $customerId = (int)$row['id_customer'];
        }
    }
    if ($customerId) {
        $cart->id_customer = $customerId;
    }
    // Direcciones: tomar la primera dirección del cliente o 0
    $id_address_delivery = 0;
    $id_address_invoice  = 0;
    if ($customerId) {
        $rowAddr = Db::getInstance()->getRow('SELECT id_address FROM ' . _DB_PREFIX_ . 'address WHERE id_customer = ' . (int)$customerId . ' AND deleted = 0 ORDER BY id_address ASC');
        if ($rowAddr) {
            $id_address_delivery = (int)$rowAddr['id_address'];
            $id_address_invoice  = (int)$rowAddr['id_address'];
        }
    }
    $cart->id_address_delivery = $id_address_delivery;
    $cart->id_address_invoice  = $id_address_invoice;
    $cart->id_currency = (int)Configuration::get('PS_CURRENCY_DEFAULT');
    $cart->id_lang     = (int)Configuration::get('PS_LANG_DEFAULT');
    $cart->id_shop     = (int)$context->shop->id;
    $cart->id_carrier  = (int)Configuration::get('PS_CARRIER_DEFAULT');
    $cart->add();
    // Agregar productos al carrito
    foreach ($cartItems as $item) {
        $idProduct   = (int)$item['product_id'];
        $idAttribute = (int)$item['attribute_id'];
        $qty         = (int)$item['qty'];
        $cart->updateQty($qty, $idProduct, $idAttribute, false, 'up', 0, null, false);
    }
    // Obtener módulo de pago
    $moduleName = $paymentMethod['module_name'] ?? '';
    if (!$moduleName) {
        $moduleName = 'ps_wirepayment';
    }
    $module = Module::getInstanceByName($moduleName);
    if (!$module) {
        throw new Exception('Módulo de pago no encontrado: ' . $moduleName);
    }
    // Determinar estado de pedido por defecto para pagos aceptados
    $orderState = (int)Configuration::get('PS_OS_PAYMENT');
    if (!$orderState) {
        $orderState = 2;
    }
    $amount = (float)$total;
    $customerSecureKey = '';
    if ($customerId) {
        $cust = new Customer($customerId);
        $customerSecureKey = $cust->secure_key;
    }
    // Validar la orden
    $module->validateOrder(
        (int)$cart->id,
        $orderState,
        $amount,
        $paymentMethod['name'] ?? $module->displayName,
        null,
        [],
        (int)$cart->id_currency,
        false,
        $customerSecureKey
    );
    $orderId = (int)$module->currentOrder;
    if (!$orderId) {
        throw new Exception('No se pudo crear el pedido en PrestaShop');
    }
    // Registrar venta en la base de datos del PDV
    $pdo = getPDO();
    if (!$pdo) {
        throw new Exception('No se pudo conectar a la base de datos del PDV');
    }
    ensureSalesTables($pdo);
    $stmtSale = $pdo->prepare('INSERT INTO sales (user_id, payment_method_id, order_id, subtotal, discount, total, created_at) VALUES (:user_id, :pm, :order_id, :subtotal, :discount, :total, NOW())');
    $stmtSale->execute([
        ':user_id'  => $userId,
        ':pm'       => (int)$paymentMethod['id'],
        ':order_id' => $orderId,
        ':subtotal' => $subtotal,
        ':discount' => $discount,
        ':total'    => $total,
    ]);
    $saleId = (int)$pdo->lastInsertId();
    $stmtItem = $pdo->prepare('INSERT INTO sales_items (sale_id, product_id, attribute_id, name, reference, quantity, unit_price) VALUES (:sale_id, :pid, :attr, :name, :ref, :qty, :price)');
    foreach ($cartItems as $item) {
        $stmtItem->execute([
            ':sale_id' => $saleId,
            ':pid'     => (int)$item['product_id'],
            ':attr'    => (int)$item['attribute_id'],
            ':name'    => $item['name'],
            ':ref'     => $item['reference'],
            ':qty'     => (int)$item['qty'],
            ':price'   => (float)$item['price'],
        ]);
    }
    return $orderId;
}

/**
 * Devuelve una conexión PDO a la base de datos de PrestaShop.
 *
 * Se utilizan los datos almacenados en el archivo de configuración generados durante la instalación
 * (host, database, username, password, prefix y charset). Devuelve null si faltan datos o la
 * conexión falla.
 *
 * @return PDO|null
 */
function getPsPDO(): ?PDO
{
    static $psPdo;
    if ($psPdo instanceof PDO) {
        return $psPdo;
    }
    $config = loadConfig();
    if (!$config || empty($config['prestashop'])) {
        return null;
    }
    $ps = $config['prestashop'];
    // Verificar campos obligatorios
    if (empty($ps['host']) || empty($ps['database']) || empty($ps['username'])) {
        return null;
    }
    $charset = $ps['charset'] ?? 'utf8mb4';
    $dsn = sprintf('mysql:host=%s;dbname=%s;charset=%s', $ps['host'], $ps['database'], $charset);
    try {
        $psPdo = new PDO($dsn, $ps['username'], $ps['password'], [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ]);
        return $psPdo;
    } catch (PDOException $e) {
        return null;
    }
}

/**
 * Obtiene un listado básico de productos desde la base de datos de PrestaShop.
 *
 * Este método lee únicamente productos simples y toma el nombre y el precio base de la tabla `product`.
 * Si se proporciona un término de búsqueda, filtrará por nombre o referencia. Limita el resultado a 50
 * productos para no sobrecargar la interfaz.
 *
 * @param string $search Término de búsqueda (opcional)
 * @return array
 */
function getProducts(string $search = ''): array
{
    // Compatibilidad: utiliza getProductVariants() para obtener productos y combinaciones.
    $variants = getProductVariants($search);
    $products = [];
    foreach ($variants as $var) {
        $products[] = [
            'id'        => $var['key'],
            'reference' => $var['reference'],
            'name'      => $var['name'],
            'price'     => $var['price'],
            'stock'     => $var['stock'],
        ];
    }
    return $products;
}

/**
 * Obtiene un producto por su ID desde la base de datos de PrestaShop.
 *
 * @param int $productId
 * @return array|null
 */
function getProductById(int $productId): ?array
{
    $psPdo = getPsPDO();
    if (!$psPdo) {
        return null;
    }
    $config = loadConfig();
    $prefix = 'ps_';
    if ($config && isset($config['prestashop']) && isset($config['prestashop']['prefix'])) {
        $prefix = $config['prestashop']['prefix'] ?: 'ps_';
    }
    $langId = 1;
    $sql = "SELECT p.id_product AS id, p.reference, pl.name, p.price\n" .
           "FROM {$prefix}product p\n" .
           "JOIN {$prefix}product_lang pl ON p.id_product = pl.id_product\n" .
           "WHERE pl.id_lang = :lang AND p.id_product = :id LIMIT 1";
    $stmt = $psPdo->prepare($sql);
    $stmt->execute([':lang' => $langId, ':id' => $productId]);
    $row = $stmt->fetch();
    if ($row) {
        return [
            'id'        => (int)$row['id'],
            'reference' => $row['reference'],
            'name'      => $row['name'],
            'price'     => (float)$row['price'],
        ];
    }
    return null;
}

/**
 * Obtiene todas las variantes disponibles (productos simples y combinaciones) de PrestaShop.
 * Cada variante tiene una clave única que identifica el producto y su combinación. Si el producto
 * no tiene combinaciones, la variante tendrá attribute_id = 0.
 *
 * Las variantes devuelven la siguiente estructura:
 *  - key: string única con formato "{product_id}-{attribute_id}" (attribute_id = 0 para simples)
 *  - product_id: ID del producto en PrestaShop
 *  - attribute_id: ID de la combinación (0 para productos sin combinación)
 *  - name: nombre del producto. Para combinaciones se añade la lista de atributos.
 *  - reference: referencia de la combinación si existe, en su defecto la del producto
 *  - price: precio base + impacto de la combinación (sin impuestos)
 *  - stock: cantidad disponible en ps_stock_available (campo quantity)
 *
 * Se realiza una sola conexión a la base de datos de PrestaShop y se ejecutan dos consultas:
 * una para obtener las combinaciones y otra para los productos simples que no tienen combinaciones.
 *
 * @param string $search Término de búsqueda opcional (filtra por nombre o referencia)
 * @return array Lista de variantes
 */
function getProductVariants(string $search = ''): array
{
    $psPdo = getPsPDO();
    if (!$psPdo) {
        return [];
    }
    $config = loadConfig();
    $prefix = 'ps_';
    if ($config && isset($config['prestashop']['prefix'])) {
        $prefix = $config['prestashop']['prefix'] ?: 'ps_';
    }
    $langId = 1;
    $variants = [];
    // Consulta para combinaciones
    // Para evitar el error HY093 (número de parámetros no válido) se utilizan nombres de parámetros únicos para cada aparición
    // de id_lang, ya que algunos drivers PDO no permiten reutilizar el mismo nombre en una consulta preparada cuando
    // ATTR_EMULATE_PREPARES está desactivado.
    $sqlComb = "SELECT pa.id_product_attribute AS attribute_id, pa.id_product, " .
        "pl.name AS product_name, p.reference AS product_ref, pa.reference AS combination_ref, " .
        "pa.price AS attribute_price, p.price AS base_price, sa.quantity, attr_names.attr_names " .
        "FROM {$prefix}product_attribute pa " .
        "JOIN {$prefix}product p ON p.id_product = pa.id_product " .
        "JOIN {$prefix}product_lang pl ON pl.id_product = p.id_product AND pl.id_lang = :lang_pl " .
        "JOIN {$prefix}stock_available sa ON sa.id_product = p.id_product AND sa.id_product_attribute = pa.id_product_attribute " .
        "JOIN (" .
        "    SELECT pac.id_product_attribute, GROUP_CONCAT(al.name ORDER BY agl.name SEPARATOR ', ') AS attr_names " .
        "    FROM {$prefix}product_attribute_combination pac " .
        "    JOIN {$prefix}attribute a ON a.id_attribute = pac.id_attribute " .
        "    JOIN {$prefix}attribute_lang al ON al.id_attribute = a.id_attribute AND al.id_lang = :lang_al " .
        "    JOIN {$prefix}attribute_group ag ON ag.id_attribute_group = a.id_attribute_group " .
        "    JOIN {$prefix}attribute_group_lang agl ON agl.id_attribute_group = ag.id_attribute_group AND agl.id_lang = :lang_ag " .
        "    GROUP BY pac.id_product_attribute" .
        ") AS attr_names ON attr_names.id_product_attribute = pa.id_product_attribute " .
        "WHERE p.active = 1";
    // Parámetros para consulta de combinaciones
    $paramsComb = [
        ':lang_pl' => $langId,
        ':lang_al' => $langId,
        ':lang_ag' => $langId,
    ];
    if ($search !== '') {
        $sqlComb .= " AND (pl.name LIKE :term_c OR p.reference LIKE :term_c OR pa.reference LIKE :term_c OR attr_names.attr_names LIKE :term_c)";
        $paramsComb[':term_c'] = '%' . $search . '%';
    }
    // Ejecutar consulta de combinaciones
    $stmtComb = $psPdo->prepare($sqlComb);
    $stmtComb->execute($paramsComb);
    while ($row = $stmtComb->fetch()) {
        // Generar clave única
        $key   = $row['id_product'] . '-' . $row['attribute_id'];
        // Nombre completo para combinaciones: nombre del producto + listado de atributos
        $name  = $row['product_name'];
        $attrs = $row['attr_names'];
        if ($attrs) {
            $name .= ' - ' . $attrs;
        }
        // Determinar referencia: usar referencia de la combinación si existe, en su defecto la del producto
        $ref = $row['combination_ref'];
        if (!$ref) {
            $ref = $row['product_ref'];
        }
        // Precio final base + impacto
        $price = (float)$row['base_price'] + (float)$row['attribute_price'];
        $variants[] = [
            'key'        => $key,
            'product_id' => (int)$row['id_product'],
            'attribute_id' => (int)$row['attribute_id'],
            'name'       => $name,
            'reference'  => $ref,
            'price'      => $price,
            'stock'      => (int)$row['quantity'],
        ];
    }
    // Consulta para productos simples sin combinaciones
    $sqlSimple = "SELECT p.id_product, pl.name AS product_name, p.reference AS product_ref, p.price AS base_price, sa.quantity " .
        "FROM {$prefix}product p " .
        "JOIN {$prefix}product_lang pl ON p.id_product = pl.id_product AND pl.id_lang = :lang " .
        "JOIN {$prefix}stock_available sa ON sa.id_product = p.id_product AND sa.id_product_attribute = 0 " .
        "WHERE p.active = 1 AND NOT EXISTS (SELECT 1 FROM {$prefix}product_attribute pa2 WHERE pa2.id_product = p.id_product)";
    $paramsSimple = [':lang' => $langId];
    if ($search !== '') {
        $sqlSimple .= " AND (pl.name LIKE :term_s OR p.reference LIKE :term_s)";
        $paramsSimple[':term_s'] = '%' . $search . '%';
    }
    $stmtSimple = $psPdo->prepare($sqlSimple);
    $stmtSimple->execute($paramsSimple);
    while ($row = $stmtSimple->fetch()) {
        $key   = $row['id_product'] . '-0';
        $variants[] = [
            'key'        => $key,
            'product_id' => (int)$row['id_product'],
            'attribute_id' => 0,
            'name'       => $row['product_name'],
            'reference'  => $row['product_ref'],
            'price'      => (float)$row['base_price'],
            'stock'      => (int)$row['quantity'],
        ];
    }
    // Ordenar por nombre para mostrar de forma coherente
    usort($variants, function ($a, $b) {
        return strcmp(mb_strtolower($a['name']), mb_strtolower($b['name']));
    });
    return $variants;
}

/**
 * Obtiene una variante concreta a partir de su clave única (product_id-attribute_id).
 * Devuelve null si no existe o no se puede recuperar.
 *
 * @param string $key Clave con formato "id_producto-id_atributo" (por ejemplo "3-0" o "5-12")
 * @return array|null
 */
function getVariant(string $key): ?array
{
    // Parsear clave
    if (!preg_match('/^(\d+)-(\d+)$/', $key, $matches)) {
        return null;
    }
    $productId   = (int)$matches[1];
    $attributeId = (int)$matches[2];
    $psPdo = getPsPDO();
    if (!$psPdo) {
        return null;
    }
    $config = loadConfig();
    $prefix = 'ps_';
    if ($config && isset($config['prestashop']['prefix'])) {
        $prefix = $config['prestashop']['prefix'] ?: 'ps_';
    }
    $langId = 1;
    if ($attributeId > 0) {
        // Variante con combinación específica
        // Usar nombres de parámetros únicos para cada aparición de id_lang para evitar error HY093
        $sql = "SELECT pa.id_product_attribute AS attribute_id, p.id_product, pl.name AS product_name, " .
            "p.reference AS product_ref, pa.reference AS combination_ref, pa.price AS attribute_price, p.price AS base_price, " .
            "sa.quantity, attr_names.attr_names " .
            "FROM {$prefix}product_attribute pa " .
            "JOIN {$prefix}product p ON p.id_product = pa.id_product " .
            "JOIN {$prefix}product_lang pl ON pl.id_product = p.id_product AND pl.id_lang = :lang_pl " .
            "JOIN {$prefix}stock_available sa ON sa.id_product = p.id_product AND sa.id_product_attribute = pa.id_product_attribute " .
            "JOIN (" .
            "    SELECT pac.id_product_attribute, GROUP_CONCAT(al.name ORDER BY agl.name SEPARATOR ', ') AS attr_names " .
            "    FROM {$prefix}product_attribute_combination pac " .
            "    JOIN {$prefix}attribute a ON a.id_attribute = pac.id_attribute " .
            "    JOIN {$prefix}attribute_lang al ON al.id_attribute = a.id_attribute AND al.id_lang = :lang_al " .
            "    JOIN {$prefix}attribute_group ag ON ag.id_attribute_group = a.id_attribute_group " .
            "    JOIN {$prefix}attribute_group_lang agl ON agl.id_attribute_group = ag.id_attribute_group AND agl.id_lang = :lang_ag " .
            "    GROUP BY pac.id_product_attribute" .
            ") AS attr_names ON attr_names.id_product_attribute = pa.id_product_attribute " .
            "WHERE pa.id_product_attribute = :attr AND pa.id_product = :pid LIMIT 1";
        $stmt = $psPdo->prepare($sql);
        // Ejecutar con parámetros únicos para cada placeholder
        $stmt->execute([
            ':lang_pl' => $langId,
            ':lang_al' => $langId,
            ':lang_ag' => $langId,
            ':attr'    => $attributeId,
            ':pid'     => $productId,
        ]);
        $row = $stmt->fetch();
        if ($row) {
            $name = $row['product_name'];
            if ($row['attr_names']) {
                $name .= ' - ' . $row['attr_names'];
            }
            $ref = $row['combination_ref'];
            if (!$ref) {
                $ref = $row['product_ref'];
            }
            $price = (float)$row['base_price'] + (float)$row['attribute_price'];
            return [
                'key'        => $productId . '-' . $attributeId,
                'product_id' => (int)$row['id_product'],
                'attribute_id' => (int)$row['attribute_id'],
                'name'       => $name,
                'reference'  => $ref,
                'price'      => $price,
                'stock'      => (int)$row['quantity'],
            ];
        }
        return null;
    }
    // Variante de producto simple sin combinaciones
    $sql = "SELECT p.id_product, pl.name AS product_name, p.reference AS product_ref, p.price AS base_price, sa.quantity " .
        "FROM {$prefix}product p " .
        "JOIN {$prefix}product_lang pl ON p.id_product = pl.id_product AND pl.id_lang = :lang " .
        "JOIN {$prefix}stock_available sa ON sa.id_product = p.id_product AND sa.id_product_attribute = 0 " .
        "WHERE p.id_product = :pid LIMIT 1";
    $stmt = $psPdo->prepare($sql);
    $stmt->execute([
        ':lang' => $langId,
        ':pid'  => $productId,
    ]);
    $row = $stmt->fetch();
    if ($row) {
        return [
            'key'        => $productId . '-0',
            'product_id' => (int)$row['id_product'],
            'attribute_id' => 0,
            'name'       => $row['product_name'],
            'reference'  => $row['product_ref'],
            'price'      => (float)$row['base_price'],
            'stock'      => (int)$row['quantity'],
        ];
    }
    return null;
}

/**
 * Obtiene todos los medios de pago.
 *
 * @param PDO $pdo
 * @return array
 */
function getPaymentMethods(PDO $pdo): array
{
    $stmt = $pdo->query('SELECT * FROM payment_methods ORDER BY id ASC');
    return $stmt->fetchAll();
}

/**
 * Obtiene los datos de un medio de pago por ID.
 *
 * @param PDO $pdo
 * @param int $id
 * @return array|null
 */
function getPaymentMethodById(PDO $pdo, int $id): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM payment_methods WHERE id = :id');
    $stmt->execute([':id' => $id]);
    $pm = $stmt->fetch();
    return $pm ?: null;
}

/**
 * Crea un nuevo medio de pago.
 *
 * @param PDO $pdo
 * @param string $name
 * @param bool $active
 * @param float $discountPercent
 */
function createPaymentMethod(PDO $pdo, string $name, bool $active, float $discountPercent, string $moduleName = ''): void
{
    // Este método ha sido simplificado: los medios de pago en el PDV ya no se asocian
    // con módulos de PrestaShop. Se ignora cualquier valor de $moduleName y se deja
    // vacío el campo module_name en la base de datos. Se conserva la columna para
    // compatibilidad con versiones anteriores, pero no se usa.
    $stmt = $pdo->prepare('INSERT INTO payment_methods (name, module_name, active, discount_percent) VALUES (:name, \'\', :active, :discount)');
    $stmt->execute([
        ':name'     => $name,
        ':active'   => $active ? 1 : 0,
        ':discount' => $discountPercent,
    ]);
}

/**
 * Actualiza un medio de pago existente.
 *
 * @param PDO $pdo
 * @param int $id
 * @param string $name
 * @param bool $active
 * @param float $discountPercent
 */
function updatePaymentMethod(PDO $pdo, int $id, string $name, bool $active, float $discountPercent, string $moduleName = ''): void
{
    // Al actualizar un medio de pago, no se modifica el valor de module_name. Se deja
    // una cadena vacía para mantener la compatibilidad, pero no se utiliza al
    // generar pedidos reales.
    $stmt = $pdo->prepare('UPDATE payment_methods SET name = :name, module_name = \'\', active = :active, discount_percent = :discount WHERE id = :id');
    $stmt->execute([
        ':name'     => $name,
        ':active'   => $active ? 1 : 0,
        ':discount' => $discountPercent,
        ':id'       => $id,
    ]);
}

/**
 * Elimina un medio de pago.
 *
 * @param PDO $pdo
 * @param int $id
 */
function deletePaymentMethod(PDO $pdo, int $id): void
{
    $stmt = $pdo->prepare('DELETE FROM payment_methods WHERE id = :id');
    $stmt->execute([':id' => $id]);
}

/**
 * Obtiene un medio de pago por su ID.
 *
 * @param PDO $pdo
 * @param int $id
 * @return array|null
 */
// Nota: La función getPaymentMethodById se declara anteriormente. Esta segunda declaración
// redundante se ha eliminado para evitar errores de redeclaración.

/**
 * Devuelve todas las cajas.
 *
 * @param PDO $pdo
 * @return array
 */
function getBoxes(PDO $pdo): array
{
    $stmt = $pdo->query('SELECT * FROM boxes ORDER BY id ASC');
    return $stmt->fetchAll();
}

/**
 * Devuelve los datos de una caja.
 *
 * @param PDO $pdo
 * @param int $id
 * @return array|null
 */
function getBoxById(PDO $pdo, int $id): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM boxes WHERE id = :id');
    $stmt->execute([':id' => $id]);
    $box = $stmt->fetch();
    return $box ?: null;
}

/**
 * Crea una nueva caja.
 *
 * @param PDO $pdo
 * @param string $name
 * @param string $description
 */
function createBox(PDO $pdo, string $name, string $description): void
{
    $stmt = $pdo->prepare('INSERT INTO boxes (name, description) VALUES (:name, :description)');
    $stmt->execute([':name' => $name, ':description' => $description]);
}

/**
 * Actualiza una caja.
 *
 * @param PDO $pdo
 * @param int $id
 * @param string $name
 * @param string $description
 */
function updateBox(PDO $pdo, int $id, string $name, string $description): void
{
    $stmt = $pdo->prepare('UPDATE boxes SET name = :name, description = :description WHERE id = :id');
    $stmt->execute([':name' => $name, ':description' => $description, ':id' => $id]);
}

/**
 * Elimina una caja si no tiene sesión abierta.
 *
 * @param PDO $pdo
 * @param int $id
 */
function deleteBox(PDO $pdo, int $id): void
{
    // Comprobar si hay sesión abierta
    $open = getOpenSession($pdo, $id);
    if ($open) {
        return;
    }
    $stmt = $pdo->prepare('DELETE FROM boxes WHERE id = :id');
    $stmt->execute([':id' => $id]);
}

/**
 * Obtiene la sesión abierta de una caja (si existe).
 *
 * @param PDO $pdo
 * @param int $boxId
 * @return array|null
 */
function getOpenSession(PDO $pdo, int $boxId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM cash_sessions WHERE box_id = :box_id AND closed_at IS NULL LIMIT 1');
    $stmt->execute([':box_id' => $boxId]);
    $session = $stmt->fetch();
    return $session ?: null;
}

/**
 * Abre una sesión de caja.
 *
 * @param PDO $pdo
 * @param int $boxId
 * @param int $userId
 * @param float $initialAmount
 */
function openBoxSession(PDO $pdo, int $boxId, int $userId, float $initialAmount): void
{
    // Verificar que no exista otra sesión abierta
    if (getOpenSession($pdo, $boxId)) {
        return;
    }
    $stmt = $pdo->prepare('INSERT INTO cash_sessions (box_id, user_id, initial_amount) VALUES (:box_id, :user_id, :initial_amount)');
    $stmt->execute([
        ':box_id'       => $boxId,
        ':user_id'      => $userId,
        ':initial_amount' => $initialAmount,
    ]);
}

/**
 * Registra un movimiento de caja (ingreso o retiro).
 *
 * @param PDO $pdo
 * @param int $sessionId
 * @param string $type 'ingreso' o 'retiro'
 * @param float $amount
 * @param string $description
 * @param int $userId
 */
function addCashMovement(PDO $pdo, int $sessionId, string $type, float $amount, string $description, int $userId): void
{
    $stmt = $pdo->prepare('INSERT INTO cash_movements (session_id, user_id, movement_type, amount, description) VALUES (:session, :user, :type, :amount, :description)');
    $stmt->execute([
        ':session'     => $sessionId,
        ':user'        => $userId,
        ':type'        => $type,
        ':amount'      => $amount,
        ':description' => $description,
    ]);
}

/**
 * Cierra una sesión de caja calculando el saldo esperado y la diferencia.
 *
 * @param PDO $pdo
 * @param int $sessionId
 * @param int $userId Usuario que cierra la caja
 * @param float $finalAmount
 */
function closeBoxSession(PDO $pdo, int $sessionId, int $userId, float $finalAmount): void
{
    // Obtener datos de la sesión
    $stmt = $pdo->prepare('SELECT * FROM cash_sessions WHERE id = :id');
    $stmt->execute([':id' => $sessionId]);
    $session = $stmt->fetch();
    if (!$session || $session['closed_at'] !== null) {
        return;
    }
    // Calcular saldo esperado
    $expected = (float)$session['initial_amount'];
    // Total ingresos y retiros
    $stmtSum = $pdo->prepare('SELECT SUM(CASE WHEN movement_type = "ingreso" THEN amount ELSE 0 END) AS ingresos, SUM(CASE WHEN movement_type = "retiro" THEN amount ELSE 0 END) AS retiros FROM cash_movements WHERE session_id = :session_id');
    $stmtSum->execute([':session_id' => $sessionId]);
    $sums = $stmtSum->fetch();
    $ingresos = (float)$sums['ingresos'];
    $retiros  = (float)$sums['retiros'];
    $expected = $expected + $ingresos - $retiros;
    $difference = $finalAmount - $expected;
    // Actualizar sesión
    $stmtClose = $pdo->prepare('UPDATE cash_sessions SET closed_at = NOW(), closed_by = :closed_by, final_amount = :final, difference = :diff WHERE id = :id');
    $stmtClose->execute([
        ':closed_by' => $userId,
        ':final'     => $finalAmount,
        ':diff'      => $difference,
        ':id'        => $sessionId,
    ]);
}

/**
 * Obtiene los movimientos de una sesión.
 *
 * @param PDO $pdo
 * @param int $sessionId
 * @return array
 */
function getSessionMovements(PDO $pdo, int $sessionId): array
{
    $stmt = $pdo->prepare('SELECT cm.*, u.username FROM cash_movements cm LEFT JOIN users u ON cm.user_id = u.id WHERE cm.session_id = :id ORDER BY cm.id ASC');
    $stmt->execute([':id' => $sessionId]);
    return $stmt->fetchAll();
}

/**
 * Obtiene resumen de una sesión de caja.
 *
 * @param PDO $pdo
 * @param int $sessionId
 * @return array|null
 */
function getSessionSummary(PDO $pdo, int $sessionId): ?array
{
    $stmt = $pdo->prepare('SELECT * FROM cash_sessions WHERE id = :id');
    $stmt->execute([':id' => $sessionId]);
    $session = $stmt->fetch();
    if (!$session) {
        return null;
    }
    // Totales
    $sumStmt = $pdo->prepare('SELECT SUM(CASE WHEN movement_type = "ingreso" THEN amount ELSE 0 END) AS ingresos, SUM(CASE WHEN movement_type = "retiro" THEN amount ELSE 0 END) AS retiros FROM cash_movements WHERE session_id = :session');
    $sumStmt->execute([':session' => $sessionId]);
    $sum = $sumStmt->fetch();
    $ingresos = (float)$sum['ingresos'];
    $retiros  = (float)$sum['retiros'];
    $expected = (float)$session['initial_amount'] + $ingresos - $retiros;
    return [
        'session' => $session,
        'ingresos' => $ingresos,
        'retiros'  => $retiros,
        'expected' => $expected,
        'difference' => $session['difference'],
    ];
}

/**
 * Obtiene el historial de sesiones cerradas para una caja.
 *
 * @param PDO $pdo
 * @param int $boxId
 * @return array
 */
function getSessionHistory(PDO $pdo, int $boxId): array
{
    $stmt = $pdo->prepare('SELECT cs.*, u.username AS opened_by, cu.username AS closed_by_name FROM cash_sessions cs LEFT JOIN users u ON cs.user_id = u.id LEFT JOIN users cu ON cs.closed_by = cu.id WHERE cs.box_id = :box_id AND cs.closed_at IS NOT NULL ORDER BY cs.id DESC');
    $stmt->execute([':box_id' => $boxId]);
    return $stmt->fetchAll();
}

/**
 * Inserta el usuario SuperAdmin en la base de datos.
 *
 * @param PDO $pdo
 * @param string $username
 * @param string $password
 */
function insertSuperAdmin(PDO $pdo, string $username, string $password): void
{
    $hash = password_hash($password, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare('INSERT INTO users (username, password, role) VALUES (:u, :p, :r)');
    $stmt->execute([
        ':u' => $username,
        ':p' => $hash,
        ':r' => 'superadmin',
    ]);
}

/**
 * Guarda una clave de configuración en la tabla settings. Si ya existe, la actualiza.
 *
 * @param PDO $pdo
 * @param string $name
 * @param string $value
 */
function saveSetting(PDO $pdo, string $name, string $value): void
{
    $stmt = $pdo->prepare('INSERT INTO settings (name, value) VALUES (:n, :v) ON DUPLICATE KEY UPDATE value = VALUES(value)');
    $stmt->execute([
        ':n' => $name,
        ':v' => $value,
    ]);
}

/**
 * Obtiene el valor de una configuración por su nombre.
 *
 * @param PDO $pdo
 * @param string $name
 * @return string|null
 */
function getSetting(PDO $pdo, string $name): ?string
{
    $stmt = $pdo->prepare('SELECT value FROM settings WHERE name = :n');
    $stmt->execute([':n' => $name]);
    $row = $stmt->fetch();
    return $row['value'] ?? null;
}

/**
 * Guarda el archivo de configuración con los parámetros proporcionados.
 *
 * @param array $config
 * @param string $path
 * @throws RuntimeException
 */
function saveConfigFile(array $config, string $path): void
{
    $export = var_export($config, true);
    $content = "<?php\nreturn " . $export . ";\n";
    if (file_put_contents($path, $content) === false) {
        throw new RuntimeException('No se pudo guardar el archivo de configuración.');
    }
    // Proteger el archivo contra lectura por el navegador (requiere .htaccess en servidores Apache)
}